// ==========================================================================
//
// = LIBRARY
//     PCIEFPGA
// 
// = FILENAME
//     pcietlpinfo.cpp
// 
// = COPYRIGHT
//     (C) Copyright 2007 Agilent Technologies
//
// ==========================================================================
#include "PCIETLPInfo.h"

// ATTENTION: Do not modify order without changing order in header file
/*static */const CPCIETlpInfoProperty CPCIETlpInfo::mPropTable[] = {
    {PKTPROP_Length,            0x00000001, 0, 10,  0},
    {PKTPROP_Reserved023,       0x00000000, 0, 2,  10},
    {PKTPROP_Attr,              0x00000000, 0, 2,  12},
    {PKTPROP_EP,                0x00000000, 0, 1,  14},
    {PKTPROP_TD,                0x00000000, 0, 1,  15},
    {PKTPROP_Reserved013,       0x00000000, 0, 4,  16},
    {PKTPROP_TC,                0x00000000, 0, 3,  20},
    {PKTPROP_Reserved017,       0x00000000, 0, 1,  23},
    {PKTPROP_Type,              0x00000004, 0, 5,  24},
    {PKTPROP_Fmt,               0x00000000, 0, 2,  29},
    {PKTPROP_Reserved007,       0x00000000, 0, 1,  31},
    {PKTPROP_1stDWBE,           0x0000000f, 1, 4,   0},
    {PKTPROP_LastDWBE,          0x00000000, 1, 4,   4},
    {PKTPROP_Tag,               0x00000000, 1, 8,   8},
    {PKTPROP_RequesterID,       0x00000000, 1,16,  16},
    {PKTPROP_CfgReserved111,        0x00000000, 2, 2,   0},
    {PKTPROP_CfgRegisterNumber,     0x00000000, 2, 6,   2},
    {PKTPROP_CfgExtRegisterNumber,  0x00000000, 2, 4,   8},
    {PKTPROP_CfgReserved107,        0x00000000, 2, 4,  12},
    {PKTPROP_CfgFunctionNumber,     0x00000000, 2, 3,  16},
    {PKTPROP_CfgDeviceNumber,       0x00000000, 2, 5,  19},
    {PKTPROP_CfgBusNumber,          0x00000000, 2, 8,  24},
    {PKTPROP_Mem64AddrHi,       0x00000000, 2,32,   0},
    {PKTPROP_Mem64AddrLo,       0x00000000, 3,32,   0},
    {PKTPROP_Mem64AddrRsvd151,  0x00000000, 3, 2,   0},
    {PKTPROP_Mem32Addr,         0x00000000, 2,32,   0},
    {PKTPROP_Mem32AddrRsvd111,  0x00000000, 2, 2,   0},
    {PKTPROP_MessageCode,       0x00000000, 1, 8,   0},
    {PKTPROP_DWord1,            0x00000000, 1, 32,   0},
    {PKTPROP_DWord2,            0x00000000, 2, 32,   0},
    {PKTPROP_DWord3,            0x00000000, 3, 32,   0},
    {PKTPROP_TLPType,           0x00000000, 0, 24,   7},
    {PKTPROP_CompReqID,         0x00000000, 2, 16, 16},
    {PKTPROP_CompCompID,        0x00000000, 1, 16, 16},
    {PKTPROP_CompStatus,        0x00000000, 1,  3, 13},
    {PKTPROP_CompByteCount,     0x00000000, 1, 12,  0},
    {PKTPROP_CompBCM,           0x00000000, 1,  1, 12},
    {PKTPROP_CompTag,           0x00000000, 2,  8,  8},
    {PKTPROP_CompRsvd117,       0x00000000, 2,  1,  7},
    {PKTPROP_CompLowerAddr,     0x00000000, 2,  7,  0},
    {PKTPROP_DWord0,            0x00000000, 0, 32,  0},
};

#define PROPTABLE_SIZE (sizeof (CPCIETlpInfo::mPropTable) / sizeof (CPCIETlpInfoProperty))

// ----------------------------------------------------------------------- 
CPCIETlpInfo::CPCIETlpInfo()
{  
}

// ----------------------------------------------------------------------- 
CPCIETlpInfo::~CPCIETlpInfo()
{
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIETlpInfo::getBitOffset(const EPCIESi& prop)
{
  return getPropertyInfo(prop).mBitOffset;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIETlpInfo::getBitSize(const EPCIESi& prop)
{
  return getPropertyInfo(prop).mBitSize;
}

// ----------------------------------------------------------------------- 
UInt32 
CPCIETlpInfo::getDefault(const EPCIESi& prop)
{
  return getPropertyInfo(prop).mDefault;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIETlpInfo::getDWNum(const EPCIESi& prop)
{
  return getPropertyInfo(prop).mDwNum;
}

//---------------------------------------------------------------------------
void 
CPCIETlpInfo::setPropVal
(
 const EPCIESi& prop,
 UInt32& currentVal,
 const UInt32& newOffsetValue 
)
{
  UInt32 mask;
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitOffset(prop);

  if(bitSize == 32)
  {
    currentVal = newOffsetValue;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    currentVal &= ~(mask << bitPos);
    currentVal |=  ((mask & newOffsetValue) << bitPos);
  }
}
//---------------------------------------------------------------------------
UInt32
CPCIETlpInfo::getPropVal
(
 const EPCIESi& prop,
 const UInt32& currentVal
)
{
  UInt32 mask;
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitOffset(prop);

  if(bitSize == 32)
  {
    return currentVal;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    return (currentVal >> bitPos) & mask;
  }
}
// ----------------------------------------------------------------------- 
CPCIETlpInfoProperty 
CPCIETlpInfo::getPropertyInfo(const EPCIESi& prop)
{
  switch(prop)
  {
    case PCIE_SI_LEN: return mPropTable[PKTPROP_Length];
    case PCIE_SI_RSVD023: return mPropTable[PKTPROP_Reserved023];
    case PCIE_SI_ATTR: return mPropTable[PKTPROP_Attr];
    case PCIE_SI_EP: return mPropTable[PKTPROP_EP];
    case PCIE_SI_TD: return mPropTable[PKTPROP_TD];
    case PCIE_SI_RSVD013: return mPropTable[PKTPROP_Reserved013];
    case PCIE_SI_TC: return mPropTable[PKTPROP_TC];
    case PCIE_SI_RSVD017: return mPropTable[PKTPROP_Reserved017];
    case PCIE_SI_TYPE: return mPropTable[PKTPROP_Type];
    case PCIE_SI_FMT: return mPropTable[PKTPROP_Fmt];
    case PCIE_SI_RSVD007: return mPropTable[PKTPROP_Reserved007];
    case PCIE_SI_1STDWBE: return mPropTable[PKTPROP_1stDWBE];
    case PCIE_SI_LASTDWBE: return mPropTable[PKTPROP_LastDWBE];
    case PCIE_SI_TAG: return mPropTable[PKTPROP_Tag];
    case PCIE_SI_REQID: return mPropTable[PKTPROP_RequesterID];
    case PCIE_SI_CFG_RSVD111: return mPropTable[PKTPROP_CfgReserved111];
    case PCIE_SI_CFG_REGNUM: return mPropTable[PKTPROP_CfgRegisterNumber];
    case PCIE_SI_CFG_EXTREGNUM: return mPropTable[PKTPROP_CfgExtRegisterNumber];
    case PCIE_SI_CFG_RSVD107: return mPropTable[PKTPROP_CfgReserved107];
    case PCIE_SI_CFG_FUNCNUM: return mPropTable[PKTPROP_CfgFunctionNumber];
    case PCIE_SI_CFG_DEVNUM: return mPropTable[PKTPROP_CfgDeviceNumber];
    case PCIE_SI_CFG_BUSNUM: return mPropTable[PKTPROP_CfgBusNumber];
    case PCIE_SI_MEM64_ADDRHI: return mPropTable[PKTPROP_Mem64AddrHi];
    case PCIE_SI_MEM64_ADDRLO: return mPropTable[PKTPROP_Mem64AddrLo];
    case PCIE_SI_MEM64_RSVD151: return mPropTable[PKTPROP_Mem64AddrRsvd151];
    case PCIE_SI_MEM32_ADDR: return mPropTable[PKTPROP_Mem32Addr];
    case PCIE_SI_MEM32_RSVD111: return mPropTable[PKTPROP_Mem32AddrRsvd111];
    case PCIE_SI_MCODE: return mPropTable[PKTPROP_MessageCode];
    case PCIE_SI_IO_ADDR: return mPropTable[PKTPROP_Mem32Addr];        
    case PCIE_SI_IO_RSVD111: return mPropTable[PKTPROP_Mem32AddrRsvd111];        
    case PCIE_SI_MSGAS_RSVD047: return mPropTable[PKTPROP_DWord1];    
    case PCIE_SI_MSGAS_RSVD087: return mPropTable[PKTPROP_DWord2];  
    case PCIE_SI_MSGAS_RSVD127: return mPropTable[PKTPROP_DWord3];    
    case PCIE_SI_MSG_BYTES08_11: return mPropTable[PKTPROP_DWord2];   
    case PCIE_SI_MSG_BYTES12_15: return mPropTable[PKTPROP_DWord3];   
    case PCIE_SI_MSGASD_RSVD047: return mPropTable[PKTPROP_DWord1];   
    case PCIE_SI_MSGASD_RSVD087: return mPropTable[PKTPROP_DWord2];   
    case PCIE_SI_MSGASD_RSVD127: return mPropTable[PKTPROP_DWord3];  
    case PCIE_SI_COMP_REQID: return mPropTable[PKTPROP_CompReqID];
    case PCIE_SI_COMP_COMPLID: return mPropTable[PKTPROP_CompCompID];
    case PCIE_SI_COMP_STATUS: return mPropTable[PKTPROP_CompStatus];
    case PCIE_SI_COMP_BYTECOUNT: return mPropTable[PKTPROP_CompByteCount];
    case PCIE_SI_COMP_BCM: return mPropTable[PKTPROP_CompBCM];
    case PCIE_SI_COMP_TAG: return mPropTable[PKTPROP_CompTag];
    case PCIE_SI_COMP_RSVD117: return mPropTable[PKTPROP_CompRsvd117];
    case PCIE_SI_COMP_LOWERADDR: return mPropTable[PKTPROP_CompLowerAddr];

    default:
          AGT_THROW("Invalid CPCIETlpInfo property");
  }
}

// ----------------------------------------------------------------------- 
CPCIETlpInfo::TLPType 
CPCIETlpInfo::TLPTypeGet(const UInt32& val)
{
  TLPType type = TLPTYPE_Unknown;
  UInt32 pktType = getPropVal(PCIE_SI_TYPE, val);
  UInt32 pktFmt  = getPropVal(PCIE_SI_FMT, val);
  

  switch(pktType)
  {
  case 0:
    if(pktFmt == 0) {type = TLPTYPE_MRd;}
    if(pktFmt == 1) {type = TLPTYPE_MRd;}
    if(pktFmt == 2) {type = TLPTYPE_MWr;}
    if(pktFmt == 3) {type = TLPTYPE_MWr;}
    break;
  case 1:
    if(pktFmt == 0) {type = TLPTYPE_MRdLk;}
    if(pktFmt == 1) {type = TLPTYPE_MRdLk;}
    break;
  case 2:
    if(pktFmt == 0) {type = TLPTYPE_IORd;}
    if(pktFmt == 2) {type = TLPTYPE_IOWr;}
    break;
  case 4:
    if(pktFmt == 0) {type = TLPTYPE_CfgRd0;}
    if(pktFmt == 2) {type = TLPTYPE_CfgWr0;}
    break;
  case 5:
    if(pktFmt == 0) {type = TLPTYPE_CfgRd1;}
    if(pktFmt == 2) {type = TLPTYPE_CfgWr1;}
    break;
  case 10:
    if(pktFmt == 0) {type = TLPTYPE_Cpl;}
    if(pktFmt == 2) {type = TLPTYPE_CplD;}
    break;
  case 11:
    if(pktFmt == 0) {type = TLPTYPE_CplLk;}
    if(pktFmt == 2) {type = TLPTYPE_CplDLk;}
    break;
  default:
    switch((pktType >> 3) & 0x3)
    {
    case 2:
      if(pktFmt == 1) {type = TLPTYPE_Msg;}
      if(pktFmt == 3) {type = TLPTYPE_MsgD;}
      break;
    case 3:
      if(pktFmt == 1) {type = TLPTYPE_MsgAS;}
      if(pktFmt == 3) {type = TLPTYPE_MsgASD;}
      break;
    default:
      type = TLPTYPE_Unknown;
    }

  }

  return type;
}

// ----------------------------------------------------------------------- 
void
CPCIETlpInfo::TLPTypeSet(UInt32& val, const TLPType& aType)
{
  UInt32 fmt = 0x0;
  UInt32 type = 0x0;

  switch(aType)
  {
  case TLPTYPE_MRd:
    fmt = 0x0;
    type = 0x0;
    break;
  case TLPTYPE_MRdLk:
    fmt = 0x0;
    type = 0x1;
    break;
  case TLPTYPE_MWr:
    fmt = 0x2;
    type = 0x0;
    break;
  case TLPTYPE_IORd:
    fmt = 0x0;
    type = 0x2;
    break;
  case TLPTYPE_IOWr:
    fmt = 0x2;
    type = 0x2;
    break;  
  case TLPTYPE_CfgRd0:
    fmt = 0x0;
    type = 0x4;
    break;  
  case TLPTYPE_CfgWr0:
    fmt = 0x2;
    type = 0x4;
    break;  
  case TLPTYPE_CfgRd1:
    fmt = 0x0;
    type = 0x5;
    break;  
  case TLPTYPE_CfgWr1:
    fmt = 0x2;
    type = 0x5;
    break;  
  case TLPTYPE_Msg:
    fmt = 0x1;
    type = 0x10;
    break;  
  case TLPTYPE_MsgD:
    fmt = 0x3;
    type = 0x10;
    break;  
  case TLPTYPE_MsgAS:
    fmt = 0x1;
    type = 0x18;
    break;  
  case TLPTYPE_MsgASD:
    fmt = 0x3;
    type = 0x18;
    break;  
  case TLPTYPE_Cpl:
    fmt = 0x0;
    type = 0xa;
    break;  
  case TLPTYPE_CplD:
    fmt = 0x2;
    type = 0xa;
    break;  
  case TLPTYPE_CplLk:
    fmt = 0x0;
    type = 0xb;
    break;  
  case TLPTYPE_CplDLk:
    fmt = 0x2;
    type = 0xb;
    break;  
  default:
    AGT_THROW("CPCIETlpInfo:: invalid packet type");
  }

  setPropVal(PCIE_SI_FMT, val, fmt);
  setPropVal(PCIE_SI_TYPE, val, type);

}

// ----------------------------------------------------------------------- 
UInt32
CPCIETlpInfo::getCompletionStatusVal(const CompletionStatus& cstatus)
{
  switch(cstatus)
  {
  case COMPSTATUS_SC: return 0x0;
  case COMPSTATUS_UR: return 0x1;
  case COMPSTATUS_CRS: return 0x2;
  case COMPSTATUS_CA: return 0x4;
  case COMPSTATUS_RSVD: return 0x7;
  default:
    AGT_THROW("CPCIETlpInfo:: invalid completion status");
  }
}

// ----------------------------------------------------------------------- 
CPCIETlpInfo::CompletionStatus
CPCIETlpInfo::resolveCompletionStatusVal(const UInt32& cval)
{
  switch(cval)
  {
  case 0x0: return COMPSTATUS_SC;
  case 0x1: return COMPSTATUS_UR;
  case 0x2: return COMPSTATUS_CRS;
  case 0x4: return COMPSTATUS_CA;
  default:
    return COMPSTATUS_RSVD;
  }
}

// Pattern Terms
// ----------------------------------------------------------------------- 
UInt8 
CPCIETlpInfo::getBitOffset(const EPCIEPattern& prop)
{
  return getPropertyInfo(prop).mBitOffset;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIETlpInfo::getBitSize(const EPCIEPattern& prop)
{
  return getPropertyInfo(prop).mBitSize;
}

// ----------------------------------------------------------------------- 
UInt8 
CPCIETlpInfo::getDWNum(const EPCIEPattern& prop)
{
  return getPropertyInfo(prop).mDwNum;
}

//---------------------------------------------------------------------------
void 
CPCIETlpInfo::setPropVal
(
 const EPCIEPattern& prop,
 UInt32& currentVal,
 const UInt32& newOffsetValue 
)
{
  UInt32 mask;
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitOffset(prop);

  if(bitSize == 32)
  {
    currentVal = newOffsetValue;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    currentVal &= ~(mask << bitPos);
    currentVal |=  ((mask & newOffsetValue) << bitPos);
  }
}
//---------------------------------------------------------------------------
UInt32
CPCIETlpInfo::getPropVal
(
 const EPCIEPattern& prop,
 const UInt32& currentVal
)
{
  UInt32 mask;
  UInt8 bitSize = getBitSize(prop);
  UInt8 bitPos = getBitOffset(prop);

  if(bitSize == 32)
  {
    return currentVal;
  }
  else
  {
    mask = ((0x1 << bitSize) - 1);
    return (currentVal >> bitPos) & mask;
  }
}

// ----------------------------------------------------------------------- 
CPCIETlpInfoProperty 
CPCIETlpInfo::getPropertyInfo(const EPCIEPattern& prop)
{
  switch(prop)
  {
    case PCIE_PATTERN_LEN: return mPropTable[PKTPROP_Length];
    case PCIE_PATTERN_RSVD023: return mPropTable[PKTPROP_Reserved023];
    case PCIE_PATTERN_ATTR: return mPropTable[PKTPROP_Attr];
    case PCIE_PATTERN_EP: return mPropTable[PKTPROP_EP];
    case PCIE_PATTERN_TD: return mPropTable[PKTPROP_TD];
    case PCIE_PATTERN_RSVD013: return mPropTable[PKTPROP_Reserved013];
    case PCIE_PATTERN_TC: return mPropTable[PKTPROP_TC];
    case PCIE_PATTERN_RSVD017: return mPropTable[PKTPROP_Reserved017];
    case PCIE_PATTERN_TYPE: return mPropTable[PKTPROP_Type];
    case PCIE_PATTERN_FMT: return mPropTable[PKTPROP_Fmt];
    case PCIE_PATTERN_RSVD007: return mPropTable[PKTPROP_Reserved007];
    case PCIE_PATTERN_1STDWBE: return mPropTable[PKTPROP_1stDWBE];
    case PCIE_PATTERN_LASTDWBE: return mPropTable[PKTPROP_LastDWBE];
    case PCIE_PATTERN_TAG: return mPropTable[PKTPROP_Tag];
    case PCIE_PATTERN_REQID: return mPropTable[PKTPROP_RequesterID];
    case PCIE_PATTERN_CFG_RSVD111: return mPropTable[PKTPROP_CfgReserved111];
    case PCIE_PATTERN_CFG_REGNUM: return mPropTable[PKTPROP_CfgRegisterNumber];
    case PCIE_PATTERN_CFG_EXTREGNUM: return mPropTable[PKTPROP_CfgExtRegisterNumber];
    case PCIE_PATTERN_CFG_RSVD107: return mPropTable[PKTPROP_CfgReserved107];
    case PCIE_PATTERN_CFG_FUNCNUM: return mPropTable[PKTPROP_CfgFunctionNumber];
    case PCIE_PATTERN_CFG_DEVNUM: return mPropTable[PKTPROP_CfgDeviceNumber];
    case PCIE_PATTERN_CFG_BUSNUM: return mPropTable[PKTPROP_CfgBusNumber];
    case PCIE_PATTERN_MEM64_ADDRHI: return mPropTable[PKTPROP_Mem64AddrHi];
    case PCIE_PATTERN_MEM64_ADDRLO: return mPropTable[PKTPROP_Mem64AddrLo];
    case PCIE_PATTERN_MEM64_RSVD151: return mPropTable[PKTPROP_Mem64AddrRsvd151];
    case PCIE_PATTERN_MEM32_ADDR: return mPropTable[PKTPROP_Mem32Addr];
    case PCIE_PATTERN_MEM32_RSVD111: return mPropTable[PKTPROP_Mem32AddrRsvd111];
    case PCIE_PATTERN_MCODE: return mPropTable[PKTPROP_MessageCode];
    case PCIE_PATTERN_IO_ADDR: return mPropTable[PKTPROP_Mem32Addr];        
    case PCIE_PATTERN_IO_RSVD111: return mPropTable[PKTPROP_Mem32AddrRsvd111];        
    case PCIE_PATTERN_MSGAS_RSVD047: return mPropTable[PKTPROP_DWord1];    
    case PCIE_PATTERN_MSGAS_RSVD087: return mPropTable[PKTPROP_DWord2];  
    case PCIE_PATTERN_MSGAS_RSVD127: return mPropTable[PKTPROP_DWord3];    
    case PCIE_PATTERN_MSG_BYTES08_11: return mPropTable[PKTPROP_DWord2];   
    case PCIE_PATTERN_MSG_BYTES12_15: return mPropTable[PKTPROP_DWord3];   
    case PCIE_PATTERN_MSGASD_RSVD047: return mPropTable[PKTPROP_DWord1];   
    case PCIE_PATTERN_MSGASD_RSVD087: return mPropTable[PKTPROP_DWord2];   
    case PCIE_PATTERN_MSGASD_RSVD127: return mPropTable[PKTPROP_DWord3];  
    case PCIE_PATTERN_COMP_REQID: return mPropTable[PKTPROP_CompReqID];
    case PCIE_PATTERN_COMP_COMPLID: return mPropTable[PKTPROP_CompCompID];
    case PCIE_PATTERN_COMP_STATUS: return mPropTable[PKTPROP_CompStatus];
    case PCIE_PATTERN_COMP_BYTECOUNT: return mPropTable[PKTPROP_CompByteCount];
    case PCIE_PATTERN_COMP_BCM: return mPropTable[PKTPROP_CompBCM];
    case PCIE_PATTERN_COMP_TAG: return mPropTable[PKTPROP_CompTag];
    case PCIE_PATTERN_COMP_RSVD117: return mPropTable[PKTPROP_CompRsvd117];
    case PCIE_PATTERN_COMP_LOWERADDR: return mPropTable[PKTPROP_CompLowerAddr];
    case PCIE_PATTERN_HEADER_DW0: return mPropTable[PKTPROP_DWord0];
    case PCIE_PATTERN_HEADER_DW1: return mPropTable[PKTPROP_DWord1];
    case PCIE_PATTERN_HEADER_DW2: return mPropTable[PKTPROP_DWord2];
    case PCIE_PATTERN_HEADER_DW3: return mPropTable[PKTPROP_DWord3];
    default:
          AGT_THROW("Invalid CPCIETlpInfo property");
  }
}
